package hsv

import (
	"fmt"
	"testing"
)

func TestBasic(t *testing.T) {
	test := fmt.Sprintf("%cname%cAlice%cage%c30%c", STX, US, RS, US, ETX)
	result := Parse(test)

	if len(result.Records) != 1 {
		t.Fatalf("Expected 1 record, got %d", len(result.Records))
	}
	if result.Records[0]["name"] != "Alice" {
		t.Errorf("Expected name=Alice, got %v", result.Records[0]["name"])
	}
	if result.Records[0]["age"] != "30" {
		t.Errorf("Expected age=30, got %v", result.Records[0]["age"])
	}
}

func TestMultipleRecords(t *testing.T) {
	test := fmt.Sprintf("%cname%cAlice%cname%cBob%c", STX, US, FS, US, ETX)
	result := Parse(test)

	if len(result.Records) != 2 {
		t.Fatalf("Expected 2 records, got %d", len(result.Records))
	}
}

func TestArrays(t *testing.T) {
	test := fmt.Sprintf("%ctags%ca%cb%cc%c", STX, US, GS, GS, ETX)
	result := Parse(test)

	if len(result.Records) != 1 {
		t.Fatalf("Expected 1 record, got %d", len(result.Records))
	}

	tags, ok := result.Records[0]["tags"].(Array)
	if !ok {
		t.Fatalf("Expected tags to be Array, got %T", result.Records[0]["tags"])
	}
	if len(tags) != 3 {
		t.Errorf("Expected 3 tags, got %d", len(tags))
	}
}

func TestHeader(t *testing.T) {
	test := fmt.Sprintf("%chsv%c1.0%ctype%cusers%cname%cAlice%c", SOH, US, RS, US, STX, US, ETX)
	result := Parse(test)

	if result.Header == nil {
		t.Fatal("Expected header, got nil")
	}
	if result.Header["hsv"] != "1.0" {
		t.Errorf("Expected hsv=1.0, got %v", result.Header["hsv"])
	}
	if result.Header["type"] != "users" {
		t.Errorf("Expected type=users, got %v", result.Header["type"])
	}
	if len(result.Records) != 1 {
		t.Errorf("Expected 1 record, got %d", len(result.Records))
	}
}

func TestNesting(t *testing.T) {
	test := fmt.Sprintf("%cuser%c%cname%cAlice%cemail%ca@b.com%c%c", STX, US, SO, US, RS, US, SI, ETX)
	result := Parse(test)

	if len(result.Records) != 1 {
		t.Fatalf("Expected 1 record, got %d", len(result.Records))
	}

	user, ok := result.Records[0]["user"].(Object)
	if !ok {
		t.Fatalf("Expected user to be Object, got %T", result.Records[0]["user"])
	}
	if user["name"] != "Alice" {
		t.Errorf("Expected name=Alice, got %v", user["name"])
	}
	if user["email"] != "a@b.com" {
		t.Errorf("Expected email=a@b.com, got %v", user["email"])
	}
}

func TestDeepNesting(t *testing.T) {
	test := fmt.Sprintf("%cdata%c%clevel1%c%clevel2%cdeep%c%c%c", STX, US, SO, US, SO, US, SI, SI, ETX)
	result := Parse(test)

	if len(result.Records) != 1 {
		t.Fatalf("Expected 1 record, got %d", len(result.Records))
	}

	data, ok := result.Records[0]["data"].(Object)
	if !ok {
		t.Fatalf("Expected data to be Object, got %T", result.Records[0]["data"])
	}

	level1, ok := data["level1"].(Object)
	if !ok {
		t.Fatalf("Expected level1 to be Object, got %T", data["level1"])
	}

	if level1["level2"] != "deep" {
		t.Errorf("Expected level2=deep, got %v", level1["level2"])
	}
}

func TestBinaryMode(t *testing.T) {
	binaryData := fmt.Sprintf("raw%cdata%chere", STX, ETX)
	test := fmt.Sprintf("%ctype%cimage%cdata%c%c%c%s%c%c%c", STX, US, RS, US, DLE, STX, binaryData, DLE, ETX, ETX)
	result := Parse(test)

	if len(result.Records) != 1 {
		t.Fatalf("Expected 1 record, got %d", len(result.Records))
	}
	if result.Records[0]["type"] != "image" {
		t.Errorf("Expected type=image, got %v", result.Records[0]["type"])
	}
	if result.Records[0]["data"] != binaryData {
		t.Errorf("Expected data=%q, got %v", binaryData, result.Records[0]["data"])
	}
}

func TestNewlines(t *testing.T) {
	test := fmt.Sprintf("%ctext%cline1\nline2\nline3%c", STX, US, ETX)
	result := Parse(test)

	if len(result.Records) != 1 {
		t.Fatalf("Expected 1 record, got %d", len(result.Records))
	}
	if result.Records[0]["text"] != "line1\nline2\nline3" {
		t.Errorf("Expected text with newlines, got %v", result.Records[0]["text"])
	}
}

func TestQuotes(t *testing.T) {
	test := fmt.Sprintf("%cmsg%cHe said \"hello\"%c", STX, US, ETX)
	result := Parse(test)

	if len(result.Records) != 1 {
		t.Fatalf("Expected 1 record, got %d", len(result.Records))
	}
	if result.Records[0]["msg"] != "He said \"hello\"" {
		t.Errorf("Expected msg with quotes, got %v", result.Records[0]["msg"])
	}
}

func TestMixedContent(t *testing.T) {
	test := fmt.Sprintf("ignored%cname%cAlice%calso ignored", STX, US, ETX)
	result := Parse(test)

	if len(result.Records) != 1 {
		t.Fatalf("Expected 1 record, got %d", len(result.Records))
	}
	if result.Records[0]["name"] != "Alice" {
		t.Errorf("Expected name=Alice, got %v", result.Records[0]["name"])
	}
}

func TestMultipleBlocks(t *testing.T) {
	test := fmt.Sprintf("%ca%c1%cjunk%cb%c2%c", STX, US, ETX, STX, US, ETX)
	result := Parse(test)

	if len(result.Records) != 2 {
		t.Fatalf("Expected 2 records, got %d", len(result.Records))
	}
}

func TestNestedArray(t *testing.T) {
	test := fmt.Sprintf("%cuser%c%cname%cAlice%ctags%cadmin%cuser%c%c", STX, US, SO, US, RS, US, GS, SI, ETX)
	result := Parse(test)

	if len(result.Records) != 1 {
		t.Fatalf("Expected 1 record, got %d", len(result.Records))
	}

	user, ok := result.Records[0]["user"].(Object)
	if !ok {
		t.Fatalf("Expected user to be Object, got %T", result.Records[0]["user"])
	}

	tags, ok := user["tags"].(Array)
	if !ok {
		t.Fatalf("Expected tags to be Array, got %T", user["tags"])
	}
	if len(tags) != 2 {
		t.Errorf("Expected 2 tags, got %d", len(tags))
	}
}
